(function($) {
    'use strict';

    let isChatOpen = false;
    let hasNewMessage = false;
    let chatHistory = [];
    let currentSessionId = '';
    let lastActivity = Date.now() / 1000; // Timestamp dalam detik
    let idleCheckInterval;
    let userData = {
        name: '',
        email: '',
        whatsapp: ''
    };

    if(currentSessionId) {
        $('.cs-bot-ai-message').show();
    } else {
        $('.cs-bot-ai-message').hide();
    }
    function initCSBotAI() {
        const floatingBtn = $('.cs-bot-ai-floating-btn');
        const chatContainer = $('.cs-bot-ai-chat-container');
        const closeBtn = $('.cs-bot-ai-close-btn');
        const messageInput = $('.cs-bot-ai-message-input');
        const sendBtn = $('.cs-bot-ai-send-btn');
        const chatMessages = $('.cs-bot-ai-chat-messages');
        const typingIndicator = $('.cs-bot-ai-typing-indicator');
        const notificationBadge = $('.cs-bot-ai-notification-badge');
        const clearBtn = $('.cs-bot-ai-clear-btn');
        const userForm = $('#cs-bot-ai-user-form');
        const chatInput = $('#cs-bot-ai-chat-input');
        const submitUserInfoBtn = $('#cs-bot-ai-submit-user-info');

        // Generate session ID
        currentSessionId = generateSessionId();

        // Initialize chat history from localStorage
        const savedHistory = localStorage.getItem('cs_bot_ai_history');
        if (savedHistory) {
            try {
                chatHistory = JSON.parse(savedHistory);
                renderChatHistory();
            } catch (e) {
                console.error('Error loading chat history:', e);
                chatHistory = [];
            }
        }

        // Check if user name exists in session
        checkUserNameSession();

        // Toggle chat
        floatingBtn.on('click', function(e) {
            e.stopPropagation();
            toggleChat();
        });

        // Close chat
        closeBtn.on('click', function(e) {
            e.stopPropagation();
            closeChat();
        });

        // Send message on enter
        messageInput.on('keypress', function(e) {
            if (e.which === 13 && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });

        // Send message on button click
        sendBtn.on('click', sendMessage);

        // Clear chat history
        clearBtn.on('click', clearHistory);

        // Submit user info
        submitUserInfoBtn.on('click', submitUserInfo);

        // Close when clicking outside
        $(document).on('click', function(e) {
            if (isChatOpen &&
                !chatContainer.is(e.target) &&
                chatContainer.has(e.target).length === 0 &&
                !floatingBtn.is(e.target) &&
                floatingBtn.has(e.target).length === 0) {
                closeChat();
            }
        });

        // Prevent closing when clicking inside chat
        chatContainer.on('click', function(e) {
            e.stopPropagation();
        });

        function generateSessionId() {
            return 'session_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
        }

        // Function to check if user name exists in session
        function checkUserNameSession() {
            $.ajax({
                url: cs_bot_ai_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'cs_bot_ai_check_user_session',
                    nonce: cs_bot_ai_ajax.nonce
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        if (response.data.has_user_name) {
                            userData.name = response.data.user_name;
                            // Hide form and show chat input
                            userForm.hide();
                            chatInput.show();
                            
                            // HANYA tambahkan welcome message jika tidak ada history
                            if (chatHistory.length === 0) {
                                const welcomeMessage = 'Halo ' + response.data.user_name + '! Ada yang bisa saya bantu?';
                                addMessage('assistant', welcomeMessage);
                            }
                        } else {
                            // Show form and hide chat input
                            userForm.show();
                            chatInput.hide();
                            
                            // HANYA tambahkan welcome message jika tidak ada history
                            if (chatHistory.length === 0) {
                                addMessage('assistant', cs_bot_ai_settings.welcome_message || 'Halo! Saya adalah asisten AI DeepSeek. Ada yang bisa saya bantu?');
                            }
                        }
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error checking user session:', error);
                    // Default: show form
                    userForm.show();
                    chatInput.hide();
                    
                    // HANYA tambahkan welcome message jika tidak ada history
                    if (chatHistory.length === 0) {
                        addMessage('assistant', cs_bot_ai_settings.welcome_message || 'Halo! Saya adalah asisten AI DeepSeek. Ada yang bisa saya bantu?');
                    }
                }
            });
        }

        function toggleChat() {
            isChatOpen = !isChatOpen;

            if (isChatOpen) {
                chatContainer.addClass('active');
                floatingBtn.hide();
                notificationBadge.hide();
                hasNewMessage = false;
                floatingBtn.css('animation', 'none');

                // Focus pada input yang sesuai
                if (userForm.is(':visible')) {
                    $('#cs-bot-ai-user-name').focus();
                } else {
                    messageInput.focus();
                }
            } else {
                chatContainer.removeClass('active');
                floatingBtn.show();
            }
        }

        function closeChat() {
            isChatOpen = false;
            chatContainer.removeClass('active');
            floatingBtn.show();
        }

        function showNotification() {
            if (!isChatOpen) {
                notificationBadge.show();
                hasNewMessage = true;
                floatingBtn.css('animation', 'cs-bot-ai-pulse 2s infinite');
            }
        }

        function scrollToBottom() {
            chatMessages.scrollTop(chatMessages[0].scrollHeight);
        }

        function renderChatHistory() {
            chatMessages.empty();

            // Hapus duplicate welcome messages
            const uniqueHistory = [];
            const seenMessages = new Set();
            
            chatHistory.forEach(message => {
                const messageKey = message.role + '_' + message.content.substring(0, 50);
                if (!seenMessages.has(messageKey)) {
                    seenMessages.add(messageKey);
                    uniqueHistory.push(message);
                }
            });
            
            chatHistory = uniqueHistory;

            if (chatHistory.length === 0) {
                // Biarkan kosong, welcome message akan ditambahkan di checkUserNameSession
                return;
            }

            chatHistory.forEach(message => {
                addMessage(message.role, message.content, message.timestamp, true);
            });

            scrollToBottom();
        }

        function addMessage(role, content, timestamp = null, skipSave = false) {
            const timestampFormatted = timestamp || new Date().toLocaleTimeString('id-ID', {
                hour: '2-digit',
                minute: '2-digit'
            });

            // ✅ BERSIHKAN CONTENT DARI SPASI DI AWAL
            let cleanContent = content.trim();
            cleanContent = cleanContent.replace(/^\s+/gm, '');

            const messageHtml = `
                <div class="cs-bot-ai-message ${role}">
                    <div class="cs-bot-ai-message-content">
                        ${cleanContent.replace(/\n/g, '<br>')}
                        <div class="cs-bot-ai-message-timestamp">${timestampFormatted}</div>
                    </div>
                </div>
            `;

            chatMessages.append(messageHtml);
            scrollToBottom();

            if (!skipSave && role !== 'system') {
                chatHistory.push({
                    role: role,
                    content: content,
                    timestamp: timestampFormatted
                });

                if (chatHistory.length > 10) {
                    chatHistory = chatHistory.slice(-10);
                }

                localStorage.setItem('cs_bot_ai_history', JSON.stringify(chatHistory));
            }

            if (role === 'assistant' && !isChatOpen) {
                showNotification();
            }
        }

        // Function untuk menyimpan nama ke session
        function setUserName(userName) {
            $.ajax({
                url: cs_bot_ai_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'cs_bot_ai_set_user_name',
                    user_name: userName,
                    nonce: cs_bot_ai_ajax.nonce
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        userData.name = userName;
                        console.log('Nama berhasil disimpan di session:', userName);
                    } else {
                        console.error('Gagal menyimpan nama di session:', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error saving name to session:', error);
                }
            });
        }

        // Function untuk memulai idle checker
        function startIdleChecker() {
            // Hapus interval sebelumnya jika ada
            if (idleCheckInterval) {
                clearInterval(idleCheckInterval);
            }
            
            // Set interval untuk check idle setiap 1 menit
            idleCheckInterval = setInterval(checkIdleStatus, 60000); // 60 detik
        }

        // Function untuk check idle status
        function checkIdleStatus() {
            const currentTime = Date.now() / 1000;
            const idleTime = currentTime - lastActivity;
            
            // Jika idle lebih dari 10 menit
            if (idleTime > 600) {
                $.ajax({
                    url: cs_bot_ai_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'cs_bot_ai_check_idle',
                        session_id: currentSessionId,
                        last_activity: lastActivity,
                        nonce: cs_bot_ai_ajax.nonce
                    },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success && response.data.is_idle) {
                            // Tambahkan pesan idle ke chat
                            addMessage('assistant', response.data.message);
                            
                            // Simpan ke database
                            saveMessageToDatabase('assistant', response.data.message);
                            
                            // Update last activity
                            lastActivity = response.data.current_time;
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('Error checking idle status:', error);
                    }
                });
            }
        }

        // Function untuk update last activity
        function updateLastActivity() {
            lastActivity = Date.now() / 1000;
        }

        function sendMessage() {
            const message = messageInput.val().trim();
            if (!message) return;

            updateLastActivity();

            messageInput.prop('disabled', true);
            sendBtn.prop('disabled', true);
            sendBtn.html('<i class="fas fa-spinner fa-spin"></i>');

            addMessage('user', message);
            messageInput.val('');
            showTyping();

            // ✅ SIMPAN PESAN USER KE DATABASE SEBELUM KIRIM KE AI
            saveMessageToDatabase('user', message);
            
            const requestData = {
                action: 'cs_bot_ai_send_message',
                message: message,
                history: chatHistory,
                session_id: currentSessionId,
                nonce: cs_bot_ai_ajax.nonce
            };

            $.ajax({
                url: cs_bot_ai_ajax.ajax_url,
                type: 'POST',
                data: requestData,
                dataType: 'json',
                success: function (response) {
                    console.log(response)
                    if (response.success) {
                        // ✅ BERSIHKAN RESPONSE DARI SPASI DI AWAL
                        let cleanResponse = response.data.response.trim();
                        
                        // Hapus spasi berulang di awal setiap baris
                        cleanResponse = cleanResponse.replace(/^\s+/gm, '');
                        
                        addMessage('assistant', cleanResponse);
                        saveMessageToDatabase('assistant', cleanResponse);

                        // Update last activity dari response
                        lastActivity = response.data.last_activity;
                        
                        // Restart idle checker
                        startIdleChecker();
                    } else {
                        const errorMsg = 'Maaf, kami mengalami kendala dalam berinteraksi.';
                        addMessage('assistant', errorMsg);
                        saveMessageToDatabase('assistant', errorMsg);
                    }
                },
                error: function (xhr, status, error) {
                    console.log(error)
                    let errorMsg = 'Maaf, terjadi kesalahan koneksi.';
                    try {
                        const response = JSON.parse(xhr.responseText);
                        if (response.data) {
                            errorMsg = response.data;
                        }
                    } catch (e) {}
                    addMessage('assistant', errorMsg);
                    saveMessageToDatabase('assistant', errorMsg);
                },
                complete: function() {
                    hideTyping();
                    messageInput.prop('disabled', false);
                    sendBtn.prop('disabled', false);
                    sendBtn.html('<i class="fas fa-paper-plane"></i>');
                    messageInput.focus();
                }
            });
        }

        function saveMessageToDatabase(role, content) {
            // ✅ BERSIHKAN CONTENT SEBELUM DISIMPAN
            let cleanContent = content.trim();
            cleanContent = cleanContent.replace(/^\s+/gm, '');
            
            console.log('Menyimpan pesan ke database:', role, cleanContent.substring(0, 50) + '...');
            
            $.ajax({
                url: cs_bot_ai_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'cs_bot_ai_save_message',
                    role: role,
                    message: cleanContent,
                    session_id: currentSessionId,
                    user_name: userData.name,
                    user_email: userData.email,
                    user_whatsapp: userData.whatsapp,
                    nonce: cs_bot_ai_ajax.nonce
                },
                dataType: 'json',
                success: function(response) {
                    if (!response.success) {
                        console.error('Gagal menyimpan pesan ke database:', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error saving message to database:', error);
                }
            });
        }

        function submitUserInfo() {
            const name = $('#cs-bot-ai-user-name').val().trim();
            const email = $('#cs-bot-ai-user-email').val().trim();
            const whatsapp = $('#cs-bot-ai-user-whatsapp').val().trim();
            $('.cs-bot-ai-chat-messages').show();
            // Validasi
            if (!name) {
                alert('Nama harus diisi');
                return;
            }

            if (!email) {
                alert('Email harus diisi');
                return;
            }

            if(!whatsapp) {
                alert('Email harus diisi');
                return;
            }

            if (email && !isValidEmail(email)) {
                alert('Format email tidak valid');
                return;
            }

            // Simpan data user
            userData = {
                name: name,
                email: email,
                whatsapp: whatsapp
            };

            // Sembunyikan form dan tampilkan input chat
            userForm.hide();
            chatInput.show();
            messageInput.focus();

            // Simpan user ke database
            saveUserToDatabase();
            setUserName(name);

            // HAPUS welcome message yang lama jika ada
            chatMessages.find('.cs-bot-ai-message.assistant').remove();
            
            // Tambahkan welcome message dengan user name
            const welcomeMessage = 'Halo Kak ' + name + '! Ada yang bisa saya bantu?';
            addMessage('assistant', welcomeMessage);
        }

        function isValidEmail(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }

        function saveUserToDatabase() {
            console.log('Menyimpan user:', userData);
            
            $.ajax({
                url: cs_bot_ai_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'cs_bot_ai_save_user',
                    name: userData.name,
                    email: userData.email,
                    whatsapp: userData.whatsapp,
                    session_id: currentSessionId,
                    nonce: cs_bot_ai_ajax.nonce
                },
                dataType: 'json',
                success: function(response) {
                    if (!response.success) {
                        console.error('Gagal menyimpan user ke database:', response.data);
                    } else {
                        console.log('User berhasil disimpan dengan ID:', response.data.user_id);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error saving user to database:', error);
                }
            });
        }

        function showTyping() {
            typingIndicator.show();
            scrollToBottom();
        }

        function hideTyping() {
            typingIndicator.hide();
        }

        function clearHistory() {
            if (!confirm('Apakah Anda yakin ingin menghapus riwayat percakapan?')) {
                return;
            }

            $.ajax({
                url: cs_bot_ai_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'cs_bot_ai_clear_history',
                    nonce: cs_bot_ai_ajax.nonce
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        chatHistory = [];
                        localStorage.removeItem('cs_bot_ai_history');
                        renderChatHistory();
                    } else {
                        alert('Gagal menghapus riwayat');
                    }
                },
                error: function() {
                    alert('Gagal menghubungi server');
                }
            });
        }

        // Event listeners untuk user activity
        $(document).on('mousemove keypress click', updateLastActivity);

        // Mulai idle checker ketika halaman load
        startIdleChecker();

        setTimeout(scrollToBottom, 100);
    }

    $(document).ready(function() {
        initCSBotAI();
    });

})(jQuery);