<?php

/**
 * Plugin Name: CS Bot AI
 * Plugin URI: https://adhifa.com
 * Description: Chat Bot AI pintar, dengan desain modern yang sangat cocok untuk menggantikan layanan CS. Memberikan respons cepat, cerdas, dan interaktif, membantu pelanggan secara otomatis dengan pengalaman percakapan yang alami.
 * Version: 1.0.0
 * Author: Adhifa Digital
 * License: GPL v2 or later
 * Text Domain: cs-bot-ai
 */

use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CS_BOT_AI_VERSION', '1.0.0');
define('CS_BOT_AI_DB_VERSION', '1.0.0');
define('CS_BOT_AI_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CS_BOT_AI_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('CS_BOT_AI_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
require_once CS_BOT_AI_PLUGIN_PATH . 'includes/class-deepseek-api.php';
require_once CS_BOT_AI_PLUGIN_PATH . 'includes/class-shortcode.php';
require_once CS_BOT_AI_PLUGIN_PATH . 'includes/class-admin-settings.php';

// Load Plugin Update Checker
require CS_BOT_AI_PLUGIN_PATH . 'includes/plugin-update-checker/plugin-update-checker.php';

$updateChecker = PucFactory::buildUpdateChecker(
    'https://project.adhifa.com/cs-bot-ai/update.json', // URL JSON update
    __FILE__,                                   // File utama plugin
    'cs-bot-ai'                                 // Slug unik plugin
);


class CS_Bot_AI
{

    private static $instance = null;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('plugins_loaded', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        add_action('wp_ajax_cs_bot_ai_export_csv', [$this, 'ajax_export_csv']);
        add_action('wp_ajax_cs_bot_ai_export_users_csv', [$this, 'ajax_export_users_csv']);
        add_action('admin_post_cs_bot_delete_user', [$this, 'admin_post_delete_user']);
    }

    public function init()
    {
        // Load text domain
        load_plugin_textdomain('cs-bot-ai', false, dirname(plugin_basename(__FILE__)) . '/languages');

        // Initialize classes
        CS_Bot_AI_Shortcode::get_instance();
        CS_Bot_AI_Admin_Settings::get_instance();
    }

    public function enqueue_scripts()
    {
        // Enqueue styles
        wp_enqueue_style(
            'cs-bot-ai-style',
            CS_BOT_AI_PLUGIN_URL . 'assets/css/style.css',
            array(),
            CS_BOT_AI_VERSION
        );

        // Enqueue scripts
        wp_enqueue_script(
            'axios',
            'https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js',
            array(),
            '1.6.0',
            true
        );

        wp_enqueue_script(
            'cs-bot-ai-script',
            CS_BOT_AI_PLUGIN_URL . 'assets/js/script.js',
            array('jquery', 'axios'),
            CS_BOT_AI_VERSION,
            true
        );

        // Localize script for AJAX
        wp_localize_script('cs-bot-ai-script', 'cs_bot_ai_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cs_bot_ai_nonce'),
            'base_url' => home_url('/')
        ));
    }

    public function enqueue_admin_scripts($hook)
    {
        if ('settings_page_cs-bot-ai-settings' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'cs-bot-ai-admin-style',
            CS_BOT_AI_PLUGIN_URL . 'assets/css/admin-style.css',
            array(),
            CS_BOT_AI_VERSION
        );
    }

    public function activate()
    {
        // Set default options
        $default_options = array(
            'api_key' => '',
            'default_model' => 'deepseek-chat',
            'temperature' => '0.7',
            'max_tokens' => '1000',
            'chat_title' => 'DeepSeek AI Assistant',
            'chat_subtitle' => 'Powered by DeepSeek API',
            'welcome_message' => 'Halo! Saya adalah asisten AI DeepSeek. Ada yang bisa saya bantu?',
            'button_position' => 'bottom-right'
        );

        if (!get_option('cs_bot_ai_settings')) {
            add_option('cs_bot_ai_settings', $default_options);
        }

        // Install database tables
        cs_bot_ai_install_database();
    }

    public function deactivate()
    {
        // Clean up if needed
    }

    // AJAX handler untuk export CSV
    public function ajax_export_csv()
    {
        check_ajax_referer('cs_bot_ai_export_nonce', 'nonce');

        global $wpdb;
        $tbl = $wpdb->prefix . 'cs_bot_chats';
        $users_tbl = $wpdb->prefix . 'cs_bot_users';

        $date = sanitize_text_field($_POST['date'] ?? date('Y-m-d'));
        $user_id = intval($_POST['user_id'] ?? 0);

        $start = $date . ' 00:00:00';
        $end = $date . ' 23:59:59';

        // Prepare query based on filters
        $where_clause = "c.created_at BETWEEN %s AND %s";
        $query_params = [$start, $end];

        if ($user_id > 0) {
            $where_clause .= " AND c.user_id = %d";
            $query_params[] = $user_id;
        }

        // Get all records for export
        $query = $wpdb->prepare(
            "SELECT c.*, u.name, u.email, u.whatsapp 
         FROM {$tbl} c 
         LEFT JOIN {$users_tbl} u ON u.id = c.user_id 
         WHERE {$where_clause} 
         ORDER BY c.created_at ASC",
            $query_params
        );
        $rows = $wpdb->get_results($query);

        // Set headers for CSV export
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=cs-bot-chat-' . $date . ($user_id ? '-user-' . $user_id : '') . '.csv');

        // Create output stream
        $output = fopen('php://output', 'w');

        // Add BOM to fix UTF-8 encoding in Excel
        fwrite($output, "\xEF\xBB\xBF");

        // Write header row
        fputcsv($output, ['id', 'session_id', 'created_at', 'sender', 'message', 'user_id', 'user_name', 'user_email', 'user_whatsapp']);

        // Write data rows
        foreach ($rows as $r) {
            // Clean message for CSV
            $clean_message = ltrim($r->message);
            $clean_message = str_replace(["\r", "\n"], "\\n", $clean_message);

            fputcsv($output, [
                $r->id,
                $r->session_id,
                $r->created_at,
                $r->sender,
                $clean_message,
                $r->user_id,
                $r->name,
                $r->email,
                $r->whatsapp
            ]);
        }

        fclose($output);
        wp_die(); // Important untuk menghentikan eksekusi
    }

    // Tambahkan AJAX handler untuk export users CSV
    public function ajax_export_users_csv()
    {
        check_ajax_referer('cs_bot_ai_export_nonce', 'nonce');

        global $wpdb;
        $users_tbl = $wpdb->prefix . 'cs_bot_users';
        $chats_tbl = $wpdb->prefix . 'cs_bot_chats';

        $search = sanitize_text_field($_POST['search'] ?? '');

        // Prepare query
        $where_clause = '';
        $query_params = [];

        if (!empty($search)) {
            $where_clause = " WHERE (u.name LIKE %s OR u.email LIKE %s OR u.whatsapp LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($search) . '%';
            $query_params = [$search_term, $search_term, $search_term];
        }

        // Get all users for export
        $query = "
        SELECT u.*, 
            COUNT(c.id) as chat_count,
            MAX(c.created_at) as last_chat_date
        FROM {$users_tbl} u 
        LEFT JOIN {$chats_tbl} c ON u.id = c.user_id
        {$where_clause}
        GROUP BY u.id 
        ORDER BY u.created_at DESC
    ";

        if (!empty($query_params)) {
            $query = $wpdb->prepare($query, $query_params);
        }

        $users = $wpdb->get_results($query);

        // Set headers for CSV export
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=cs-bot-users-' . date('Y-m-d') . '.csv');

        // Create output stream
        $output = fopen('php://output', 'w');

        // Add BOM to fix UTF-8 encoding in Excel
        fwrite($output, "\xEF\xBB\xBF");

        // Write header row
        fputcsv($output, [
            'ID',
            'Nama',
            'Email',
            'WhatsApp',
            'Tanggal Daftar',
            'Jumlah Chat',
            'Chat Terakhir',
            'Tanggal Update'
        ]);

        // Write data rows
        foreach ($users as $user) {
            fputcsv($output, [
                $user->id,
                $user->name,
                $user->email,
                $user->whatsapp,
                $user->created_at,
                $user->chat_count,
                $user->last_chat_date ?: '-',
                $user->updated_at
            ]);
        }

        fclose($output);
        wp_die();
    }

    // Admin post handler untuk delete user
    public function admin_post_delete_user()
    {
        global $wpdb;
        $users_tbl = $wpdb->prefix . 'cs_bot_users';
        $chats_tbl = $wpdb->prefix . 'cs_bot_chats';

        if (!isset($_POST['user_id']) || !isset($_POST['_wpnonce'])) {
            wp_die('Invalid request');
        }

        $user_id = intval($_POST['user_id']);

        // Verify nonce
        if (!wp_verify_nonce($_POST['_wpnonce'], 'delete_user_' . $user_id)) {
            wp_die('Security check failed');
        }

        // Start transaction
        $wpdb->query('START TRANSACTION');

        try {
            // First delete all chats by this user
            $wpdb->delete($chats_tbl, ['user_id' => $user_id], ['%d']);

            // Then delete the user
            $deleted = $wpdb->delete($users_tbl, ['id' => $user_id], ['%d']);

            if ($deleted) {
                $wpdb->query('COMMIT');
                wp_redirect(add_query_arg([
                    'page' => 'cs-bot-ai-users',
                    'delete_success' => 1
                ], admin_url('admin.php')));
                exit;
            } else {
                throw new Exception('User not found or already deleted');
            }
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            wp_redirect(add_query_arg([
                'page' => 'cs-bot-ai-users',
                'delete_error' => urlencode($e->getMessage())
            ], admin_url('admin.php')));
            exit;
        }
    }
}

// Initialize the plugin
CS_Bot_AI::get_instance();

// Fungsi untuk install database
function cs_bot_ai_install_database()
{
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    $tbl_users = $wpdb->prefix . 'cs_bot_users';
    $tbl_chats = $wpdb->prefix . 'cs_bot_chats';

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

    // Cek dan buat tabel users
    if ($wpdb->get_var("SHOW TABLES LIKE '{$tbl_users}'") != $tbl_users) {
        $sql = "CREATE TABLE {$tbl_users} (
            id BIGINT(20) NOT NULL AUTO_INCREMENT,
            name VARCHAR(191) DEFAULT '' NOT NULL,
            email VARCHAR(191) DEFAULT '' NOT NULL,
            whatsapp VARCHAR(50) DEFAULT '' NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id)
        ) {$charset_collate};";
        dbDelta($sql);
    }

    // Cek dan buat tabel chats
    if ($wpdb->get_var("SHOW TABLES LIKE '{$tbl_chats}'") != $tbl_chats) {
        $sql2 = "CREATE TABLE {$tbl_chats} (
            id BIGINT(20) NOT NULL AUTO_INCREMENT,
            user_id BIGINT(20) DEFAULT 0 NOT NULL,
            message LONGTEXT,
            sender VARCHAR(20) DEFAULT 'user',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            emailed TINYINT(1) DEFAULT 0,
            session_id VARCHAR(191) DEFAULT '' NOT NULL,
            PRIMARY KEY  (id)
        ) {$charset_collate};";
        dbDelta($sql2);
    }

    // Update versi database
    update_option('cs_bot_ai_db_version', CS_BOT_AI_DB_VERSION);
}

// ✅ Tambahkan auto cek lisensi di admin_init
add_action('admin_init', 'cs_bot_ai_auto_check_license');

function cs_bot_ai_auto_check_license()
{
    $license = get_option('cs_bot_ai_license_status');

    // Kalau belum ada data lisensi → skip
    if (empty($license) || empty($license['license_key'])) {
        return;
    }

    // Cek expired date dulu
    if (!empty($license['expires']) && strtotime($license['expires']) < time()) {
        update_option('cs_bot_ai_license_status', [
            'status'      => 'inactive',
            'email'       => $license['email'] ?? '',
            'license_key' => $license['license_key'],
            'expires'     => $license['expires'],
        ]);
        return;
    }

    // 🔎 Validasi ulang ke server lisensi
    $check = cs_bot_ai_validate_license($license['license_key']);

    if (!empty($check['success'])) {
        // Kalau valid, update status tetap active
        update_option('cs_bot_ai_license_status', [
            'status'      => 'active',
            'email'       => $license['email'] ?? '',
            'license_key' => $license['license_key'],
            'expires'     => $check['expires'] ?? null,
            'data'        => $check['data'] ?? [],
        ]);
    } else {
        // Kalau gagal, set inactive
        update_option('cs_bot_ai_license_status', [
            'status'      => 'inactive',
            'email'       => $license['email'] ?? '',
            'license_key' => $license['license_key'],
            'expires'     => $license['expires'] ?? null,
        ]);
    }
}

// Function untuk menyimpan nama ke session
add_action('wp_ajax_cs_bot_ai_set_user_name', 'cs_bot_ai_set_user_name');
add_action('wp_ajax_nopriv_cs_bot_ai_set_user_name', 'cs_bot_ai_set_user_name');
function cs_bot_ai_set_user_name()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');

    if (!session_id()) {
        session_start();
    }

    $user_name = sanitize_text_field($_POST['user_name'] ?? '');

    if (!empty($user_name)) {
        $_SESSION['cs_bot_user_name'] = $user_name;
        wp_send_json_success(['message' => 'Nama berhasil disimpan']);
    } else {
        wp_send_json_error('Nama tidak boleh kosong');
    }
}

// Function untuk check user session
add_action('wp_ajax_cs_bot_ai_check_user_session', 'cs_bot_ai_check_user_session');
add_action('wp_ajax_nopriv_cs_bot_ai_check_user_session', 'cs_bot_ai_check_user_session');
function cs_bot_ai_check_user_session()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');

    if (!session_id()) {
        session_start();
    }

    $has_user_name = !empty($_SESSION['cs_bot_user_name']);
    $user_name = $_SESSION['cs_bot_user_name'] ?? '';

    wp_send_json_success([
        'has_user_name' => $has_user_name,
        'user_name' => $user_name
    ]);
}


// Tambahkan hook AJAX untuk menyimpan user
add_action('wp_ajax_cs_bot_ai_save_user', 'cs_bot_ai_save_user');
add_action('wp_ajax_nopriv_cs_bot_ai_save_user', 'cs_bot_ai_save_user');

function cs_bot_ai_save_user()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');

    $name = sanitize_text_field($_POST['name'] ?? '');
    $email = sanitize_email($_POST['email'] ?? '');
    $whatsapp = sanitize_text_field($_POST['whatsapp'] ?? '');
    $session_id = sanitize_text_field($_POST['session_id'] ?? '');

    if (empty($name) || empty($email)) {
        wp_send_json_error('Nama dan email harus diisi');
    }

    global $wpdb;
    $tbl_users = $wpdb->prefix . 'cs_bot_users';

    // Debug: Log data yang diterima
    error_log('CS Bot AI - Save User Data: ' . print_r([
        'name' => $name,
        'email' => $email,
        'whatsapp' => $whatsapp,
        'session_id' => $session_id
    ], true));

    // Cek apakah user sudah ada berdasarkan email
    $existing_user = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$tbl_users} WHERE email = %s",
        $email
    ));

    if ($existing_user) {
        // Update user yang sudah ada (HANYA kolom yang ada di tabel)
        $result = $wpdb->update(
            $tbl_users,
            [
                'name' => $name,
                'whatsapp' => $whatsapp
                // session_id tidak ada di tabel users, jadi dihapus
            ],
            ['email' => $email],
            ['%s', '%s'], // format untuk values
            ['%s']        // format untuk where clause
        );

        $user_id = $existing_user->id;

        // Debug: Log hasil update
        if ($result === false) {
            error_log('CS Bot AI - Update User Error: ' . $wpdb->last_error);
        } else {
            error_log('CS Bot AI - User Updated: ' . $user_id);
        }
    } else {
        // Buat user baru (HANYA kolom yang ada di tabel)
        $result = $wpdb->insert(
            $tbl_users,
            [
                'name' => $name,
                'email' => $email,
                'whatsapp' => $whatsapp,
                'created_at' => current_time('mysql')
                // session_id tidak ada di tabel users, jadi dihapus
            ],
            ['%s', '%s', '%s', '%s'] // format untuk values
        );

        $user_id = $wpdb->insert_id;

        // Debug: Log hasil insert
        if ($result === false) {
            error_log('CS Bot AI - Insert User Error: ' . $wpdb->last_error);
        } else {
            error_log('CS Bot AI - User Inserted: ' . $user_id);
        }
    }

    if ($result === false) {
        error_log('CS Bot AI - Final Save Error: ' . $wpdb->last_error);
        wp_send_json_error('Gagal menyimpan data user: ' . $wpdb->last_error);
    }

    // Simpan session_id ke tabel chats sebagai referensi
    if (!empty($session_id)) {
        $tbl_chats = $wpdb->prefix . 'cs_bot_chats';
        $wpdb->insert(
            $tbl_chats,
            [
                'user_id' => $user_id,
                'message' => 'User registered: ' . $name . ' (' . $email . ')',
                'sender' => 'system',
                'session_id' => $session_id,
                'created_at' => current_time('mysql')
            ],
            ['%d', '%s', '%s', '%s', '%s']
        );
    }

    wp_send_json_success([
        'user_id' => $user_id,
        'message' => 'Data user berhasil disimpan'
    ]);
}

// Update fungsi save message untuk include user data

// Tambahkan hook AJAX untuk menyimpan pesan
add_action('wp_ajax_cs_bot_ai_save_message', 'cs_bot_ai_save_message');
add_action('wp_ajax_nopriv_cs_bot_ai_save_message', 'cs_bot_ai_save_message');

function cs_bot_ai_save_message()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');

    $role = sanitize_text_field($_POST['role'] ?? '');
    $message = wp_kses_post($_POST['message'] ?? '');
    $session_id = sanitize_text_field($_POST['session_id'] ?? '');
    $user_name = sanitize_text_field($_POST['user_name'] ?? '');
    $user_email = sanitize_email($_POST['user_email'] ?? '');
    $user_whatsapp = sanitize_text_field($_POST['user_whatsapp'] ?? '');

    if (empty($message) || empty($role)) {
        wp_send_json_error('Data tidak lengkap');
    }

    global $wpdb;
    $tbl_chats = $wpdb->prefix . 'cs_bot_chats';
    $tbl_users = $wpdb->prefix . 'cs_bot_users';

    // Cari user_id berdasarkan email
    $user_id = 0;
    if (!empty($user_email)) {
        $user = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$tbl_users} WHERE email = %s",
            $user_email
        ));

        if ($user) {
            $user_id = $user->id;
        }
    }

    // Simpan pesan ke database
    $result = $wpdb->insert($tbl_chats, [
        'user_id' => $user_id,
        'message' => $message,
        'sender' => $role,
        'session_id' => $session_id,
        'created_at' => current_time('mysql')
    ]);

    if ($result === false) {
        wp_send_json_error('Gagal menyimpan pesan ke database');
    }

    wp_send_json_success(['message_id' => $wpdb->insert_id]);
}

add_action('wp_ajax_cs_bot_ai_send_message', 'cs_bot_ai_send_message');
add_action('wp_ajax_nopriv_cs_bot_ai_send_message', 'cs_bot_ai_send_message');
add_action('wp_ajax_cs_bot_ai_clear_history', 'cs_bot_ai_clear_history');
add_action('wp_ajax_nopriv_cs_bot_ai_clear_history', 'cs_bot_ai_clear_history');

function cs_bot_ai_send_message()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');

    // Validasi pesan
    $message = $_POST['message'] ?? '';
    $session_id = $_POST['session_id'] ?? '';
    $last_activity = $_POST['last_activity'] ?? time(); // Tambahkan last_activity

    if (empty(trim($message))) {
        wp_send_json_error('Pesan tidak boleh kosong');
    }

    $message = sanitize_text_field($message);
    $session_id = sanitize_text_field($session_id);
    $last_activity = intval($last_activity);

    // ========== CEK IDLE TIME ==========
    $current_time = time();
    $idle_time = $current_time - $last_activity;
    $is_idle = $idle_time > 600; // 10 menit = 600 detik

    // Ambil nama dari session
    if (!session_id()) {
        session_start();
    }

    $user_name = $_SESSION['cs_bot_user_name'] ?? '';

    if (empty($user_name) && !empty($_POST['user_name'])) {
        $user_name = sanitize_text_field($_POST['user_name']);
        $_SESSION['cs_bot_user_name'] = $user_name;
    }

    if (empty($user_name)) {
        $user_name = 'Pengguna';
    }
    // ========== END CEK IDLE TIME ==========

    // Ambil history
    $history = [];
    if (!empty($_POST['history'])) {
        $decoded = is_string($_POST['history']) ? json_decode(wp_unslash($_POST['history']), true) : $_POST['history'];
        $history = is_array($decoded) ? $decoded : [];
    }

    // Ambil settings plugin
    $options = get_option('cs_bot_ai_settings', []);
    $provider = $options['provider'] ?? 'instan-crm';
    $system_message = $options['system_message'] ?? 'Anda adalah asisten AI yang membantu dan ramah.';
    $instruction = $options['instruction'] ?? '';

    // Modifikasi system message dengan nama pengguna dan instruksi idle
    $enhanced_system_message = $system_message . " Nama pengguna Anda adalah: " . $user_name .
        ". Gunakan nama tersebut saat berinteraksi untuk membuat percakapan lebih personal." .
        " Jika pengguna tidak merespon selama 10 menit, tanyakan apakah mereka masih ada dengan menyebutkan namanya.";

    // Tentukan API key, URL, model
    if ($provider == 'instan-crm') {
        $api_key = $options['instan_crm_api_key'] ?? '';
        $url     = 'https://app.instansender.com/api/openai';
    } elseif ($provider == 'instanwa') {
        $api_key = $options['instanwa_api_key'] ?? '';
        $url     = 'https://instanwa.com/api/openai';
    } elseif ($provider === 'deepseek') {
        $api_key = $options['deepseek_api_key'] ?? '';
        $url     = 'https://api.deepseek.com/v1/chat/completions';
        $model   = $options['default_model'] ?? 'deepseek-chat';
    } else {
        $api_key = $options['openai_api_key'] ?? '';
        $url     = 'https://api.openai.com/v1/chat/completions';
        $model   = $options['default_model'] ?? 'gpt-4o-mini';
    }

    if (empty($api_key)) {
        wp_send_json_error('API key tidak dikonfigurasi.');
    }

    // Siapkan messages untuk dikirim ke AI
    $messages = [
        ['role' => 'system', 'content' => $enhanced_system_message]
    ];

    if (!empty($instruction)) {
        $messages[] = ['role' => 'system', 'content' => $instruction];
    }

    if (!empty($history)) {
        foreach ($history as $h) {
            if (isset($h['role'], $h['content'])) {
                $messages[] = [
                    'role' => in_array($h['role'], ['user', 'assistant', 'system']) ? $h['role'] : 'user',
                    'content' => sanitize_text_field($h['content'])
                ];
            }
        }
    }

    $messages[] = ['role' => 'user', 'content' => $message];

    // Kirim request ke API
    if ($provider == 'instan-crm' || $provider == 'instanwa') {
        $response = wp_remote_post($url, [
            'headers' => [
                'Content-Type' => 'application/json; charset=utf-8',
            ],
            'body' => wp_json_encode([
                'token' => $api_key,
                'messages' => $messages
            ]),
            'timeout' => 30,
        ]);
    } else {
        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode([
                'model' => $model,
                'messages' => $messages,
            ]),
            'timeout' => 30,
        ]);
    }

    if (is_wp_error($response)) {
        wp_send_json_error('Koneksi gagal: ' . $response->get_error_message());
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);

    if (!isset($data['choices'][0]['message']['content'])) {
        wp_send_json_error('Tidak ada respons dari AI');
    }

    $ai_response = wp_kses_post($data['choices'][0]['message']['content']);

    // HAPUS spasi di awal dan akhir response
    $ai_response = trim($ai_response);
    $ai_response = str_replace('**', '', $ai_response);
    $ai_response = str_replace('###', '', $ai_response);

    // Tambahkan line break setelah tanda titik, tanda seru, atau tanda tanya
    $ai_response = preg_replace('/([.!?])\s+(?![0-9])/', "$1\n", $ai_response);

    // Tambahkan line break setelah titik dua jika bukan URL atau angka
    $ai_response = preg_replace('/:(?!\/\/)(?![0-9])\s*/', ":\n", $ai_response);

    // ✅ UBAH URL MENJADI CLICKABLE LINK
    $ai_response = make_clickable($ai_response);

    // ========== TAMBAHKAN PESAN IDLE JIKA PERLU ==========
    if ($is_idle && !empty($user_name) && $user_name !== 'Pengguna') {
        $idle_message = "\n\n---\n*Halo " . esc_html($user_name) . ", apakah Anda masih ada? Silakan balas jika Anda masih membutuhkan bantuan.*";
        $ai_response .= $idle_message;
    }
    // ========== END PESAN IDLE ==========

    // Update history
    $history[] = ['role' => 'user', 'content' => $message];
    $history[] = ['role' => 'assistant', 'content' => $ai_response];

    wp_send_json_success([
        'response' => $ai_response,
        'history'  => $history,
        'user_name' => $user_name,
        'last_activity' => $current_time, // Kirim waktu activity terbaru
        'is_idle' => $is_idle // Flag untuk frontend
    ]);
}

function cs_bot_ai_clear_history()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');
    wp_send_json_success(['history' => []]);
}

// Function untuk check idle dan kirim reminder
add_action('wp_ajax_cs_bot_ai_check_idle', 'cs_bot_ai_check_idle');
add_action('wp_ajax_nopriv_cs_bot_ai_check_idle', 'cs_bot_ai_check_idle');
function cs_bot_ai_check_idle()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');

    if (!session_id()) {
        session_start();
    }

    $session_id = sanitize_text_field($_POST['session_id'] ?? '');
    $last_activity = intval($_POST['last_activity'] ?? 0);
    $current_time = time();

    $user_name = $_SESSION['cs_bot_user_name'] ?? 'Pengguna';

    // Cek jika idle lebih dari 10 menit
    if (($current_time - $last_activity) > 600) {
        $idle_message = "Halo " . esc_html($user_name) . ", apakah Anda masih ada? Silakan balas jika Anda masih membutuhkan bantuan.";

        wp_send_json_success([
            'is_idle' => true,
            'message' => $idle_message,
            'current_time' => $current_time
        ]);
    }

    wp_send_json_success([
        'is_idle' => false,
        'current_time' => $current_time
    ]);
}


// Tambahkan hook untuk AJAX
add_action('wp_ajax_cs_bot_ai_activate_license', 'ajax_activate_license');
add_action('wp_ajax_nopriv_cs_bot_ai_activate_license', 'ajax_activate_license');

function ajax_activate_license()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');

    $license_key = sanitize_text_field($_POST['license_key'] ?? '');
    // Bisa datang sebagai 'email' atau 'license_email'
    $email       = sanitize_email($_POST['email'] ?? $_POST['license_email'] ?? get_option('admin_email'));

    if (empty($license_key)) {
        wp_send_json_error(['message' => 'License key kosong.']);
    }

    // 1) Cek lisensi dulu (GET /cek-lisensi)
    $validasi = cs_bot_ai_validate_license($license_key);
    if (!empty($validasi['success'])) {
        update_option('cs_bot_ai_license_status', [
            'status'      => 'active',
            'email'       => $email,
            'license_key' => $license_key,
            'expires'     => $validasi['expires'] ?? null,
            'data'        => $validasi['data'] ?? [],
        ]);
        wp_send_json_success(['message' => $validasi['message'] ?? 'Lisensi valid. Plugin aktif.']);
    }

    // 2) Kalau belum valid → aktivasi (POST /aktivasi-lisensi)
    $result = cs_bot_ai_check_license($email, $license_key);

    if (!empty($result['success'])) {
        update_option('cs_bot_ai_license_status', [
            'status'      => 'active',
            'email'       => $email,
            'license_key' => $license_key,
            'expires'     => $result['expires'] ?? null,
            'data'        => $result['data'] ?? [],
        ]);
        wp_send_json_success(['message' => $result['message'] ?? 'Lisensi berhasil diaktifkan.']);
    } else {
        update_option('cs_bot_ai_license_status', [
            'status'      => 'inactive',
            'email'       => $email,
            'license_key' => $license_key,
        ]);
        wp_send_json_error(['message' => $result['message'] ?? 'Lisensi tidak valid. Silakan daftar.']);
    }
}

function cs_bot_ai_check_license($email, $serial)
{
    $domain = home_url();
    $url    = 'https://lisensi.adhifa.com/aktivasi-lisensi';

    $response = wp_remote_post($url, [
        'timeout'   => 30,
        'sslverify' => true,        // set true di production
        'body'      => [
            'email'  => sanitize_email($email),
            'serial' => sanitize_text_field($serial),
            'domain' => $domain,
        ],
    ]);

    if (is_wp_error($response)) {
        error_log('Lisensi API Error: ' . $response->get_error_message());
        return ['success' => false, 'message' => 'Koneksi ke server lisensi gagal.'];
    }

    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    if ($code !== 200 || empty($body)) {
        error_log('Lisensi API HTTP ' . $code . ' Body: ' . $body);
        return ['success' => false, 'message' => 'License server tidak merespons (HTTP ' . $code . ').'];
    }

    $data = json_decode($body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return ['success' => false, 'message' => 'Response JSON tidak valid dari server lisensi.'];
    }

    // Contoh success payload:
    // { status: "success", message: "...", data: { serial, domain, expired, version, link_download } }
    if (!empty($data['status']) && $data['status'] === 'success') {
        update_option('cs_bot_ai_license_key', $serial);
        update_option('cs_bot_ai_license_data', $data['data'] ?? []);
        return [
            'success' => true,
            'message' => $data['message'] ?? 'Lisensi diaktivasi.',
            'expires' => $data['data']['expired'] ?? null,
            'data'    => $data['data'] ?? [],
        ];
    }

    return [
        'success' => false,
        'message' => $data['message'] ?? 'Lisensi tidak valid.',
    ];
}


function cs_bot_ai_validate_license($serial)
{
    $domain = home_url();

    $url = add_query_arg(
        [
            'serial' => $serial,   // biarkan mentah, add_query_arg yang encode
            'domain' => $domain,
        ],
        'https://lisensi.adhifa.com/cek-lisensi'
    );

    $response = wp_remote_get($url, [
        'timeout'   => 30,
        'sslverify' => true,
    ]);

    if (is_wp_error($response)) {
        error_log('Lisensi API Error: ' . $response->get_error_message());
        return ['success' => false, 'message' => $response->get_error_message()];
    }

    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    if ($code !== 200 || empty($body)) {
        error_log('Lisensi API Invalid response: HTTP ' . $code);
        return ['success' => false, 'message' => 'Invalid response (HTTP ' . $code . ').'];
    }

    $data = json_decode($body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return ['success' => false, 'message' => 'Response JSON tidak valid.'];
    }

    // Sesuaikan dengan contoh payload yang Anda kirim
    if (!empty($data['status']) && $data['status'] === 'success') {
        update_option('cs_bot_ai_license_data', $data['data'] ?? []);
        return [
            'success' => true,
            'message' => $data['message'] ?? 'Lisensi valid.',
            'expires' => $data['data']['expired'] ?? null,
            'data'    => $data['data'] ?? [],
        ];
    }

    return ['success' => false, 'message' => $data['message'] ?? 'Lisensi tidak valid.'];
}

add_action('wp_ajax_cs_bot_ai_deactivate_license', 'ajax_deactivate_license');
function ajax_deactivate_license()
{
    check_ajax_referer('cs_bot_ai_nonce', 'nonce');
    delete_option('cs_bot_ai_license_status');
    wp_send_json_success(['message' => 'Lisensi berhasil dinonaktifkan.']);
}


// Enqueue admin script
add_action('admin_enqueue_scripts', function ($hook) {
    // Hanya load di halaman plugin kita
    if (strpos($hook, 'cs-bot-ai') === false) {
        return;
    }

    // Daftarkan JS
    wp_enqueue_script(
        'cs-bot-ai-admin',
        plugin_dir_url(__FILE__) . 'assets/js/admin.js',
        array('jquery'),
        '1.0',
        true
    );

    // Localize script agar JS bisa akses ajax_url
    wp_localize_script('cs-bot-ai-admin', 'cs_bot_ai_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('cs_bot_ai_nonce')
    ));
});


// Tambahkan link "Pengaturan" di daftar plugin
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'cs_bot_ai_settings_link');

function cs_bot_ai_settings_link($links)
{
    $settings_link = '<a href="admin.php?page=cs-bot-ai-settings">Pengaturan</a>';
    array_unshift($links, $settings_link); // taruh paling depan
    return $links;
}
