<?php
if (!defined('ABSPATH')) {
    exit;
}

class CS_Bot_AI_Admin_Settings
{
    private static $instance = null;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));

        // Validasi settings
        add_filter('pre_update_option_cs_bot_ai_settings', array($this, 'validate_settings'), 10, 2);
    }

    public function validate_settings($new_value, $old_value)
    {
        // Normalisasi auto_show
        if (!isset($new_value['auto_show'])) {
            $new_value['auto_show'] = false;
        } else {
            $new_value['auto_show'] = (bool) $new_value['auto_show'];
        }

        return $new_value;
    }

    // Tambahkan di dalam class CS_Bot_AI_Admin_Settings
    public function add_admin_menu()
    {
        // Menu utama di sidebar
        add_menu_page(
            'CS Bot AI',              // Page title
            'CS Bot AI',              // Menu title
            'manage_options',         // Capability
            'cs-bot-ai',              // Menu slug
            array($this, 'main_page'), // Callback function
            'dashicons-format-chat',  // Icon menu
            25                        // Position
        );

        if ($this->cs_bot_ai_is_license_active()) {
            // Sub menu 1: Riwayat Chat
            add_submenu_page(
                'cs-bot-ai',                   // Parent slug
                'User Chat',                // Page title
                'User Chat',                // Menu title
                'manage_options',              // Capability
                'cs-bot-ai-users',      // Menu slug
                array($this, 'daftar_user')   // Callback function
            );

            add_submenu_page(
                'cs-bot-ai',
                'Riwayat Chat',
                'Riwayat Chat',
                'manage_options',
                'cs-bot-ai-history',
                array($this, 'riwayat_chat')
            );

            // Sub menu 2: Tampilan
            add_submenu_page(
                'cs-bot-ai',
                'Tampilan',
                'Tampilan',
                'manage_options',
                'cs-bot-ai-tampilan',
                array($this, 'tampilan_page')
            );
        }

        // Sub menu 3: Pengaturan
        add_submenu_page(
            'cs-bot-ai',
            'Pengaturan',
            'Pengaturan',
            'manage_options',
            'cs-bot-ai-settings',
            array($this, 'pengaturan_page')
        );
    }

    public function settings_init()
    {
        register_setting('cs_bot_ai', 'cs_bot_ai_settings');

        add_settings_field(
            'license_email',
            'License Email',
            array($this, 'license_email_render'),
            'cs_bot_ai',
            'cs_bot_ai_license_section'
        );

        // Tambahkan field license hanya jika belum aktif
        if (!$this->cs_bot_ai_is_license_active()) {
            // Section License (selalu muncul)
            add_settings_section(
                'cs_bot_ai_license_section',
                'License',
                function () {
                    echo '<p>Masukkan license key untuk mengaktifkan plugin.</p>';
                },
                'cs_bot_ai'
            );

            add_settings_field(
                'license_key',
                'License Key',
                array($this, 'license_key_render'),
                'cs_bot_ai',
                'cs_bot_ai_license_section'
            );
        }

        // Section API & Chat hanya jika lisensi aktif
        if ($this->cs_bot_ai_is_license_active()) {
            // API Settings
            add_settings_section(
                'cs_bot_ai_api_section',
                'API Settings',
                array($this, 'api_section_callback'),
                'cs_bot_ai'
            );

            add_settings_field(
                'provider',
                'AI Provider',
                array($this, 'provider_render'),
                'cs_bot_ai',
                'cs_bot_ai_api_section'
            );

            add_settings_field(
                'instan_crm_api_key',
                'Instan CRM API Key',
                array($this, 'instan_crm_api_key_render'),
                'cs_bot_ai',
                'cs_bot_ai_api_section',
                array('class' => 'row-instan-crm-key')
            );

            add_settings_field(
                'instanwa_api_key',
                'InstanWA API Key',
                array($this, 'instanwa_api_key_render'),
                'cs_bot_ai',
                'cs_bot_ai_api_section',
                array('class' => 'row-instanwa-key')
            );

            add_settings_field(
                'deepseek_api_key',
                'DeepSeek API Key',
                array($this, 'deepseek_api_key_render'),
                'cs_bot_ai',
                'cs_bot_ai_api_section',
                array('class' => 'row-deepseek-key')
            );

            add_settings_field(
                'openai_api_key',
                'OpenAI API Key',
                array($this, 'openai_api_key_render'),
                'cs_bot_ai',
                'cs_bot_ai_api_section',
                array('class' => 'row-openai-key')
            );

            // Chat Settings
            add_settings_section(
                'cs_bot_ai_chat_section',
                'Chat Settings',
                array($this, 'chat_section_callback'),
                'cs_bot_ai'
            );

            add_settings_field(
                'chat_title',
                'Chat Title',
                array($this, 'chat_title_render'),
                'cs_bot_ai',
                'cs_bot_ai_chat_section'
            );

            add_settings_field(
                'chat_subtitle',
                'Chat Subtitle',
                array($this, 'chat_subtitle_render'),
                'cs_bot_ai',
                'cs_bot_ai_chat_section'
            );

            add_settings_field(
                'system_message',
                'System Message',
                array($this, 'system_message_render'),
                'cs_bot_ai',
                'cs_bot_ai_chat_section'
            );

            add_settings_field(
                'instruction',
                'More Instruction',
                array($this, 'instruction_render'),
                'cs_bot_ai',
                'cs_bot_ai_chat_section'
            );

            add_settings_field(
                'auto_show',
                'Auto Show Chat',
                array($this, 'auto_show_render'),
                'cs_bot_ai',
                'cs_bot_ai_chat_section'
            );
        }
    }

    /* ================= RENDER FIELD ================= */

    public function render_license_section()
    {
        $license_status = get_option('cs_bot_ai_license_status', []);
        $status = $license_status['status'] ?? 'inactive';
        $email  = $license_status['email'] ?? '';
        $key    = $license_status['license_key'] ?? '';
        $exp    = $license_status['expires'] ?? '';

        if ($status === 'active') {
            echo '<p><strong style="color:green;">✅ Lisensi aktif</strong></p>';
            if (!empty($email)) {
                echo '<p>Email: <code>' . esc_html($email) . '</code></p>';
            }
            if (!empty($key)) {
                echo '<p>License Key: <code>' . esc_html($key) . '</code></p>';
            }
            if (!empty($exp)) {
                echo '<p>Berlaku sampai: <code>' . esc_html($exp) . '</code></p>';
            }

            echo '<p><button type="button" class="button button-secondary" id="cs-bot-ai-deactivate-license">Deactivate License</button></p>';
            echo '<div id="cs-bot-ai-license-msg"></div>';
        } else {
            echo '<p>Masukkan license key untuk mengaktifkan plugin.</p>';
            echo '<table class="form-table">
                <tr>
                    <th scope="row"><label for="license_email">License Email</label></th>
                    <td><input type="email" name="cs_bot_ai_settings[license_email]" value="' . esc_attr($email) . '" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="license_key">License Key</label></th>
                    <td><input type="text" name="cs_bot_ai_settings[license_key]" value="' . esc_attr($key) . '" class="regular-text" /></td>
                </tr>
            </table>
            <p><button type="button" class="button button-primary" id="cs-bot-ai-activate-license">Activate License</button></p>
            <div id="cs-bot-ai-license-msg"></div>';
        }
    }


    public function license_email_render()
    {
        $options = get_option('cs_bot_ai_settings');
?>
        <input type="email" name="cs_bot_ai_settings[license_email]"
            value="<?php echo esc_attr($options['license_email'] ?? ''); ?>"
            class="regular-text">
        <p class="description">Masukkan email yang terdaftar untuk lisensi plugin.</p>
    <?php
    }


    public function license_key_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <input type="text" name="cs_bot_ai_settings[license_key]"
            value="<?php echo esc_attr($options['license_key'] ?? ''); ?>"
            class="regular-text">
        <p class="description">Masukkan license key untuk mengaktifkan plugin.</p>
    <?php
    }

    public function provider_render()
    {
        $options = get_option('cs_bot_ai_settings');
        $provider = $options['provider'] ?? 'instan-crm';
    ?>
        <select name="cs_bot_ai_settings[provider]">
            <option value="instan-crm" <?php selected($provider, 'instan-crm'); ?>>Instan CRM</option>
            <option value="instanwa" <?php selected($provider, 'instanwa'); ?>>InstanWA</option>
            <!-- <option value="deepseek" <?php selected($provider, 'deepseek'); ?>>DeepSeek</option>
            <option value="openai" <?php selected($provider, 'openai'); ?>>OpenAI</option> -->
        </select>
        <p class="description">Pilih penyedia AI.</p>
    <?php
    }

    public function instan_crm_api_key_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <input type="password" name="cs_bot_ai_settings[instan_crm_api_key]"
            value="<?php echo esc_attr($options['instan_crm_api_key'] ?? ''); ?>"
            class="regular-text">
        <p class="description">Masukkan API key dari
            <a href="https://instansender.com" target="_blank">Instan CRM</a>.
        </p>
    <?php
    }

    public function instanwa_api_key_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <input type="password" name="cs_bot_ai_settings[instanwa_api_key]"
            value="<?php echo esc_attr($options['instanwa_api_key'] ?? ''); ?>"
            class="regular-text">
        <p class="description">Masukkan API key dari
            <a href="https://instanwa.com" target="_blank">InstanWA</a>.
        </p>
    <?php
    }

    public function deepseek_api_key_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <input type="password" name="cs_bot_ai_settings[deepseek_api_key]"
            value="<?php echo esc_attr($options['deepseek_api_key'] ?? ''); ?>"
            class="regular-text">
        <p class="description">Masukkan API key dari
            <a href="https://platform.deepseek.com" target="_blank">DeepSeek</a>.
        </p>
    <?php
    }

    public function openai_api_key_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <input type="password" name="cs_bot_ai_settings[openai_api_key]"
            value="<?php echo esc_attr($options['openai_api_key'] ?? ''); ?>"
            class="regular-text">
        <p class="description">Masukkan API key dari
            <a href="https://platform.openai.com" target="_blank">OpenAI</a>.
        </p>
    <?php
    }

    public function chat_title_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <input type="text" name="cs_bot_ai_settings[chat_title]"
            value="<?php echo esc_attr($options['chat_title'] ?? 'Live Chat'); ?>"
            class="regular-text">
    <?php
    }

    public function chat_subtitle_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <input type="text" name="cs_bot_ai_settings[chat_subtitle]"
            value="<?php echo esc_attr($options['chat_subtitle'] ?? 'Silakan Chat Dengan CS Kami'); ?>"
            class="regular-text">
    <?php
    }

    public function system_message_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <textarea name="cs_bot_ai_settings[system_message]" rows="3" class="large-text"><?php
                                                                                        echo esc_textarea($options['system_message'] ?? 'Anda adalah asisten AI yang membantu dan ramah.');
                                                                                        ?></textarea>
        <p class="description">Pesan ini akan dijadikan role system (definisi dasar AI).</p>
    <?php
    }

    public function instruction_render()
    {
        $options = get_option('cs_bot_ai_settings');
    ?>
        <textarea name="cs_bot_ai_settings[instruction]" rows="3" class="large-text"><?php
                                                                                        echo esc_textarea($options['instruction'] ?? 'Kamu adalah asisten AI yang ramah dan membantu.');
                                                                                        ?></textarea>
        <p class="description">Instruksi ini akan selalu dikirim ke AI sebagai perintah dasar.</p>
    <?php
    }

    public function auto_show_render()
    {
        $options = get_option('cs_bot_ai_settings');
        $auto_show = isset($options['auto_show']) ? $options['auto_show'] : true;
    ?>
        <label>
            <input type="checkbox" name="cs_bot_ai_settings[auto_show]" value="1" <?php checked($auto_show, true); ?>>
            Tampilkan chat otomatis di semua halaman
        </label>
    <?php
    }

    public function api_section_callback()
    {
        echo '<p>Konfigurasi API settings untuk DeepSeek atau OpenAI</p>';
    }

    public function chat_section_callback()
    {
        echo '<p>Konfigurasi tampilan chat bot</p>';
    }

    /* ================= OPTIONS PAGE ================= */

    public function pengaturan_page()
    {
        $is_license_active = $this->cs_bot_ai_is_license_active();
    ?>
        <div class="wrap">
            <h1>CS Bot AI Settings</h1>
            <form action="options.php" method="post">
                <?php
                settings_fields('cs_bot_ai');
                do_settings_sections('cs_bot_ai');
                submit_button();
                ?>
            </form>

            <?php if (!$is_license_active): ?>
                <!-- License Activation -->
                <div style="margin-top: 20px;">
                    <button type="button" class="button" id="cs-bot-ai-activate-license">Aktifkan Lisensi</button>
                    <span id="cs-bot-ai-license-msg" style="margin-left:10px;"></span>
                </div>


                <p style="margin-top:15px;color:red;">
                    Masukkan license key dan klik "Aktifkan Lisensi" untuk membuka pengaturan API & Chat.<br>
                    Jika belum memiliki lisensi, silakan <a href="https://app.adhifa.com/checkout/cs-bot-ai" target="_blank">Klik disini</a>
                </p>
            <?php endif; ?>

            <?php if ($is_license_active): ?>
                <!-- 📌 Shortcode Info -->
                <div class="cs-bot-ai-shortcode" style="margin-top:30px;padding:15px;border:1px solid #ccd0d4;background:#fff;">
                    <h2>📌 Cara Menampilkan Chat Bot</h2>
                    <p>Anda dapat menampilkan chat bot di halaman atau postingan menggunakan shortcode berikut:</p>
                    <ul style="list-style:disc;margin-left:20px;">
                        <li>
                            <code>[cs_bot_ai]</code><br>
                            <span class="description">Tampilkan chat bot dengan posisi default (pojok kanan bawah).</span>
                        </li>
                        <li style="margin-top:8px;">
                            <code>[cs_bot_ai position="bottom-left"]</code><br>
                            <span class="description">Tampilkan chat bot di pojok kiri bawah.</span>
                        </li>
                    </ul>
                    <p>Shortcode ini bisa digunakan di <strong>halaman, postingan, atau widget</strong> yang mendukung shortcode.</p>
                </div>
            <?php endif; ?>
            <script>
                jQuery(document).ready(function($) {
                    function toggleApiFields() {
                        var provider = $('select[name="cs_bot_ai_settings[provider]"]').val();
                        if (provider === 'instan-crm') {
                            $('.row-instan-crm-key').show();
                            $('.row-instanwa-key').hide();
                            $('.row-deepseek-key').hide();
                            $('.row-openai-key').hide();
                        } else if (provider === 'instanwa') {
                            $('.row-instan-crm-key').hide();
                            $('.row-instanwa-key').show();
                            $('.row-deepseek-key').hide();
                            $('.row-openai-key').hide();
                        } else if (provider === 'deepseek') {
                            $('.row-instan-crm-key').hide();
                            $('.row-instanwa-key').hide();
                            $('.row-deepseek-key').show();
                            $('.row-openai-key').hide();
                        } else {
                            $('.row-instan-crm-key').hide();
                            $('.row-instanwa-key').hide();
                            $('.row-deepseek-key').hide();
                            $('.row-openai-key').show();
                        }
                    }

                    toggleApiFields();
                    $('select[name="cs_bot_ai_settings[provider]"]').on('change', toggleApiFields);

                    // License activation
                    $('#cs-bot-ai-activate-license').on('click', function() {
                        var licenseKey = $('input[name="cs_bot_ai_settings[license_key]"]').val();
                        var licenseEmail = $('input[name="cs_bot_ai_settings[license_email]"]').val();

                        var data = {
                            action: 'cs_bot_ai_activate_license',
                            license_key: licenseKey,
                            license_email: licenseEmail,
                            nonce: cs_bot_ai_ajax.nonce
                        };

                        $.post(cs_bot_ai_ajax.ajax_url, data, function(response) {
                            $('#cs-bot-ai-license-msg').text(response.data || response);
                            if (response.success) {
                                $('#cs-bot-ai-license-msg').css('color', 'green');
                                location.reload();
                            } else {
                                $('#cs-bot-ai-license-msg').css('color', 'red');
                            }
                        }, 'json');
                    });

                });
            </script>
        </div>
    <?php
    }

    /* ================= LICENSE HELPER ================= */

    public function cs_bot_ai_is_license_active()
    {
        $license = get_option('cs_bot_ai_license_status');
        return !empty($license) && $license['status'] === 'active';
    }

    // Contoh AJAX handler kirim pesan
    public function cs_bot_ai_send_message()
    {
        check_ajax_referer('cs_bot_ai_nonce', 'nonce');

        if (!$this->cs_bot_ai_is_license_active()) {
            wp_send_json_error('Plugin belum diaktifkan. Masukkan license key terlebih dahulu.');
        }

        // ...lanjutkan proses kirim pesan ke AI
    }

    // Callback untuk halaman utama
    public function main_page()
    {
        echo '
    <div class="wrap">
        <h1>Selamat datang di CS Bot AI</h1>
        
        <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
            <h2 style="color: #2c3e50; margin-top: 0;">Tentang PT Adhifa Cipta Digitama</h2>
            
            <div style="display: flex; flex-wrap: wrap; gap: 20px; align-items: center;">
                <div style="flex: 1; min-width: 300px;">
                    <p style="font-size: 16px; line-height: 1.6; color: #34495e;">
                        <strong>PT Adhifa Cipta Digitama</strong> adalah perusahaan yang bergerak di bidang 
                        teknologi informasi dan pengembangan software. Kami berkomitmen untuk memberikan 
                        solusi digital yang inovatif dan berkualitas tinggi untuk membantu bisnis Anda 
                        berkembang di era digital.
                    </p>
                    
                    <div style="background: white; padding: 15px; border-radius: 6px; margin: 15px 0;">
                        <h3 style="color: #e74c3c; margin-top: 0;">📌 Informasi Kontak</h3>
                        <p style="margin: 8px 0;">
                            <strong>📍 Alamat:</strong><br>
                            Jl. Pokoh Gg. Nakulo No 139 Pokoh,<br>
                            Wedomartani, Ngemplak,<br>
                            Sleman, Yogyakarta
                        </p>
                        <p style="margin: 8px 0;">
                            <strong>🌐 Website:</strong><br>
                            <a href="https://adhifa.com" target="_blank" style="color: #3498db; text-decoration: none;">
                                https://adhifa.com
                            </a>
                        </p>
                    </div>
                </div>
                
                <div style="flex: 1; min-width: 300px; text-align: center;">
                    <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
                                color: white; padding: 30px; border-radius: 10px;">
                        <h3 style="text-align: left; margin-bottom: 20px; padding-left: 20px; margin-top: 0; color: #fff;">🚀 Layanan Kami</h3>
                        <ul style="text-align: left; margin: 0; padding-left: 20px;">
                            <li>✅ Pengembangan Website & Aplikasi</li>
                            <li>✅ Digital Marketing Solutions</li>
                            <li>✅ AI & Chatbot Integration</li>
                            <li>✅ Consulting IT</li>
                            <li>✅ Maintenance & Support</li>
                        </ul>

                        <div style="margin-top: 30px; margin-left: 20px; text-align: left;">
                            <a href="https://adhifa.com/kontak-kami" target="_blank" 
                            style="background: #27ae60; color: white; padding: 12px 25px; 
                                    text-decoration: none; border-radius: 5px; display: inline-block;">
                                📞 Hubungi Kami
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            
            <div style="margin-top: 25px; padding: 15px; background: #e8f4f8; border-radius: 6px;">
                <h3 style="color: #2980b9; margin-top: 0;">💡 Tentang CS Bot AI</h3>
                <p style="margin: 0; color: #2c3e50;">
                    Plugin ini dikembangkan oleh PT Adhifa Cipta Digitama untuk memberikan 
                    layanan customer service yang efisien dan responsif melalui teknologi 
                    Artificial Intelligence.
                </p>
            </div>
        </div>
        
        
    </div>
    
    <style>
        .wrap {
            max-width: 1200px;
            margin: 0 auto;
        }
        a:hover {
            opacity: 0.9;
        }
        @media (max-width: 768px) {
            .wrap {
                padding: 10px;
            }
        }
    </style>
    ';
    }

    public function daftar_user()
    {
        global $wpdb;
        $users_tbl = $wpdb->prefix . 'cs_bot_users';
        $chats_tbl = $wpdb->prefix . 'cs_bot_chats';

        // Handle delete action
        if (isset($_GET['action']) && $_GET['action'] === 'delete_user' && isset($_GET['user_id'])) {
            $this->handle_delete_user();
        }

        // Pagination
        $per_page = 20;
        $current_page = max(1, intval($_GET['paged'] ?? 1));
        $offset = ($current_page - 1) * $per_page;

        // Search functionality
        $search = sanitize_text_field($_GET['s'] ?? '');
        $where_clause = '';
        $query_params = [];

        if (!empty($search)) {
            $where_clause = " WHERE (u.name LIKE %s OR u.email LIKE %s OR u.whatsapp LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($search) . '%';
            $query_params = [$search_term, $search_term, $search_term];
        }

        // Get total users for pagination
        $count_query = "SELECT COUNT(*) FROM {$users_tbl} u {$where_clause}";
        if (!empty($query_params)) {
            $count_query = $wpdb->prepare($count_query, $query_params);
        }
        $total_users = $wpdb->get_var($count_query);
        $total_pages = ceil($total_users / $per_page);

        // Get users with chat count
        $users_query = "
        SELECT u.*, 
            COUNT(c.id) as chat_count,
            MAX(c.created_at) as last_chat_date
        FROM {$users_tbl} u 
        LEFT JOIN {$chats_tbl} c ON u.id = c.user_id
        {$where_clause}
        GROUP BY u.id 
        ORDER BY u.created_at DESC 
        LIMIT %d OFFSET %d
    ";

        $final_query_params = array_merge($query_params, [$per_page, $offset]);
        $users = $wpdb->get_results($wpdb->prepare($users_query, $final_query_params));

    ?>
        <div class="wrap">
            <h1>Daftar User CS Bot AI</h1>

            <?php if (isset($_GET['delete_success'])): ?>
                <div class="notice notice-success is-dismissible">
                    <p>User berhasil dihapus!</p>
                </div>
            <?php endif; ?>

            <?php if (isset($_GET['delete_error'])): ?>
                <div class="notice notice-error is-dismissible">
                    <p>Gagal menghapus user: <?php echo esc_html($_GET['delete_error']); ?></p>
                </div>
            <?php endif; ?>

            <!-- Search Form -->
            <form method="get" style="margin-bottom: 20px;">
                <input type="hidden" name="page" value="cs-bot-ai-users" />
                <label>
                    Cari User:
                    <input type="text" name="s" value="<?php echo esc_attr($search); ?>"
                        placeholder="Nama, Email, atau WhatsApp" style="width: 300px;">
                </label>
                <button type="submit" class="button">Cari</button>
                <?php if (!empty($search)): ?>
                    <a href="?page=cs-bot-ai-users" class="button">Reset</a>
                <?php endif; ?>

                <!-- Export CSV Button -->
                <button type="button" class="button button-primary" id="cs-bot-export-users-csv"
                    data-search="<?php echo esc_attr($search); ?>">
                    Export CSV
                </button>
            </form>

            <!-- Pagination Info -->
            <div style="margin: 15px 0;">
                <strong>Total: <?php echo number_format($total_users); ?> user</strong>
                <?php if ($total_pages > 1): ?>
                    <span style="margin-left: 15px;">
                        Halaman <?php echo $current_page; ?> dari <?php echo $total_pages; ?>
                    </span>
                <?php endif; ?>
            </div>

            <!-- Users Table -->
            <table class="widefat fixed striped" style="margin-top: 10px;">
                <thead>
                    <tr>
                        <th width="5%">No</th>
                        <th width="15%">Nama</th>
                        <th width="20%">Email</th>
                        <th width="15%">WhatsApp</th>
                        <th width="15%">Tanggal Daftar</th>
                        <th width="10%">Jumlah Chat</th>
                        <th width="15%">Chat Terakhir</th>
                        <th width="15%">Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($users)): ?>
                        <tr>
                            <td colspan="8" style="text-align: center;">
                                <?php echo empty($search) ? 'Belum ada user terdaftar' : 'User tidak ditemukan'; ?>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php
                        $start_number = $offset + 1;
                        foreach ($users as $user): ?>
                            <tr>
                                <td><?php echo $start_number++; ?></td>
                                <td><?php echo esc_html($user->name); ?></td>
                                <td><?php echo esc_html($user->email); ?></td>
                                <td><?php echo esc_html($user->whatsapp); ?></td>
                                <td><?php echo esc_html(date('d M Y H:i', strtotime($user->created_at))); ?></td>
                                <td><?php echo esc_html($user->chat_count); ?></td>
                                <td>
                                    <?php if ($user->last_chat_date): ?>
                                        <?php echo esc_html(date('d M Y H:i', strtotime($user->last_chat_date))); ?>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div style="display: flex; gap: 5px; flex-wrap: wrap;">
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=cs-bot-ai-history&user_id=' . $user->id)); ?>"
                                            class="button button-small" title="Lihat Chat">
                                            👁️
                                        </a>
                                        <a href="<?php echo esc_url(wp_nonce_url(
                                                        add_query_arg([
                                                            'page' => 'cs-bot-ai-users',
                                                            'action' => 'delete_user',
                                                            'user_id' => $user->id
                                                        ], admin_url('admin.php')),
                                                        'delete_user_' . $user->id
                                                    )); ?>"
                                            class="button button-small button-link-delete"
                                            title="Hapus User"
                                            onclick="return confirm('Apakah Anda yakin ingin menghapus user <?php echo esc_js($user->name); ?>? Semua chat yang terkait juga akan dihapus.')">
                                            🗑️
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <!-- Pagination Navigation -->
            <?php if ($total_pages > 1): ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <?php
                        $base_url = add_query_arg([
                            'page' => 'cs-bot-ai-users',
                            's' => $search
                        ], admin_url('admin.php'));

                        echo paginate_links([
                            'base' => $base_url . '%_%',
                            'format' => '&paged=%#%',
                            'current' => $current_page,
                            'total' => $total_pages,
                            'prev_text' => '&laquo; Sebelumnya',
                            'next_text' => 'Berikutnya &raquo;',
                            'mid_size' => 2,
                            'end_size' => 1
                        ]);
                        ?>
                    </div>
                </div>
            <?php endif; ?>

            <script>
                jQuery(document).ready(function($) {
                    $('#cs-bot-export-users-csv').on('click', function() {
                        var search = $(this).data('search');

                        // Show loading
                        $(this).html('<span class="spinner is-active"></span> Exporting...');
                        $(this).prop('disabled', true);

                        // Create form untuk download
                        var form = $('<form>', {
                            method: 'POST',
                            action: '<?php echo admin_url("admin-ajax.php"); ?>'
                        });

                        form.append($('<input>', {
                            type: 'hidden',
                            name: 'action',
                            value: 'cs_bot_ai_export_users_csv'
                        }));

                        form.append($('<input>', {
                            type: 'hidden',
                            name: 'search',
                            value: search
                        }));

                        form.append($('<input>', {
                            type: 'hidden',
                            name: 'nonce',
                            value: '<?php echo wp_create_nonce("cs_bot_ai_export_nonce"); ?>'
                        }));

                        // Tambahkan form ke body dan submit
                        form.appendTo('body').submit().remove();

                        // Reset button setelah 2 detik
                        setTimeout(function() {
                            $('#cs-bot-export-users-csv').html('Export CSV').prop('disabled', false);
                        }, 2000);
                    });

                    // Confirm delete dengan sweetalert jika available
                    $('.button-link-delete').on('click', function(e) {
                        if (!confirm($(this).attr('onclick').replace('return confirm(', '').replace(')', '').replace(/'/g, ''))) {
                            e.preventDefault();
                        }
                    });
                });
            </script>

            <style>
                .tablenav {
                    margin: 20px 0;
                    display: flex;
                    justify-content: flex-end;
                }

                .tablenav-pages {
                    text-align: right;
                }

                .page-numbers {
                    padding: 5px 10px;
                    margin: 0 2px;
                    border: 1px solid #ccc;
                    text-decoration: none;
                }

                .page-numbers.current {
                    background: #0073aa;
                    color: white;
                    border-color: #0073aa;
                }

                .page-numbers:hover {
                    background: #f1f1f1;
                }

                .button-link-delete {
                    color: #dc3232 !important;
                    border-color: #dc3232 !important;
                    background: transparent !important;
                }

                .button-link-delete:hover {
                    background: #dc3232 !important;
                    color: white !important;
                }
            </style>
        </div>
    <?php
    }

    // Function to handle user deletion
    private function handle_delete_user()
    {
        global $wpdb;
        $users_tbl = $wpdb->prefix . 'cs_bot_users';
        $chats_tbl = $wpdb->prefix . 'cs_bot_chats';

        $user_id = intval($_GET['user_id']);

        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'delete_user_' . $user_id)) {
            wp_die('Security check failed');
        }

        // Start transaction
        $wpdb->query('START TRANSACTION');

        try {
            // First delete all chats by this user
            $wpdb->delete($chats_tbl, ['user_id' => $user_id], ['%d']);

            // Then delete the user
            $deleted = $wpdb->delete($users_tbl, ['id' => $user_id], ['%d']);

            if ($deleted) {
                $wpdb->query('COMMIT');

                // Gunakan JavaScript redirect untuk menghindari header issues
                echo '<script>window.location.href = "' . add_query_arg([
                    'page' => 'cs-bot-ai-users',
                    'delete_success' => 1
                ], admin_url('admin.php')) . '";</script>';
                exit;
            } else {
                throw new Exception('User not found or already deleted');
            }
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            echo '<script>window.location.href = "' . add_query_arg([
                'page' => 'cs-bot-ai-users',
                'delete_error' => urlencode($e->getMessage())
            ], admin_url('admin.php')) . '";</script>';
            exit;
        }
    }

    // Modifikasi fungsi riwayat_chat untuk filter by user_id
    public function riwayat_chat()
    {
        global $wpdb;
        $tbl = $wpdb->prefix . 'cs_bot_chats';
        $users_tbl = $wpdb->prefix . 'cs_bot_users';

        $date = sanitize_text_field($_GET['date'] ?? date('Y-m-d'));
        $user_id = intval($_GET['user_id'] ?? 0);
        $export = isset($_GET['export']) ? true : false;

        // Handle export first - before any output
        if ($export) {
            $this->export_chat_csv($date, $user_id);
            exit;
        }

        // Pagination parameters
        $per_page = 20;
        $current_page = max(1, intval($_GET['paged'] ?? 1));
        $offset = ($current_page - 1) * $per_page;

        $start = $date . ' 00:00:00';
        $end = $date . ' 23:59:59';

        // Prepare query based on filters
        $where_clause = "c.created_at BETWEEN %s AND %s";
        $query_params = [$start, $end];

        if ($user_id > 0) {
            $where_clause .= " AND c.user_id = %d";
            $query_params[] = $user_id;
        }

        // Get total count for pagination
        $count_query = $wpdb->prepare(
            "SELECT COUNT(*) 
         FROM {$tbl} c 
         LEFT JOIN {$users_tbl} u ON u.id = c.user_id 
         WHERE {$where_clause}",
            $query_params
        );
        $total_items = $wpdb->get_var($count_query);
        $total_pages = ceil($total_items / $per_page);

        // Main query with pagination
        $query = $wpdb->prepare(
            "SELECT c.*, u.name, u.email, u.whatsapp 
         FROM {$tbl} c 
         LEFT JOIN {$users_tbl} u ON u.id = c.user_id 
         WHERE {$where_clause} 
         ORDER BY c.created_at DESC 
         LIMIT %d OFFSET %d",
            array_merge($query_params, [$per_page, $offset])
        );

        $rows = $wpdb->get_results($query);

        // Get user info if filtering by user
        $user_info = '';
        if ($user_id > 0) {
            $user = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$users_tbl} WHERE id = %d",
                $user_id
            ));
            if ($user) {
                $user_info = " - " . esc_html($user->name . " (" . $user->email . ")");
            }
        }

    ?>
        <div class="wrap">
            <h1>Riwayat Chat<?php echo $user_info; ?> - <?php echo esc_html($date); ?></h1>

            <form method="get">
                <input type="hidden" name="page" value="cs-bot-ai-history" />
                <?php if ($user_id > 0): ?>
                    <input type="hidden" name="user_id" value="<?php echo esc_attr($user_id); ?>" />
                <?php endif; ?>

                <label>Pilih tanggal:
                    <input type="date" name="date" value="<?php echo esc_attr($date); ?>" />
                </label>

                <button class="button">Filter</button>

                <!-- Ganti link export dengan button yang trigger AJAX -->
                <button type="button" class="utton button-primary" id="cs-bot-export-csv"
                    data-date="<?php echo esc_attr($date); ?>"
                    data-user-id="<?php echo esc_attr($user_id); ?>">
                    Export CSV
                </button>

                <?php if ($user_id > 0): ?>
                    <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=cs-bot-ai-users')); ?>">
                        Kembali ke Daftar User
                    </a>
                <?php endif; ?>
            </form>

            <!-- Pagination Info -->
            <div style="margin: 15px 0;">
                <strong>Menampilkan <?php echo number_format($total_items); ?> data</strong>
                <?php if ($total_pages > 1): ?>
                    <span style="margin-left: 15px;">
                        Halaman <?php echo $current_page; ?> dari <?php echo $total_pages; ?>
                    </span>
                <?php endif; ?>
            </div>

            <table class="widefat fixed striped" style="margin-top: 10px;">
                <thead>
                    <tr>
                        <th width="5%">No</th>
                        <th width="15%">Tanggal</th>
                        <th width="10%">Sender</th>
                        <th width="45%">Message</th>
                        <th width="25%">User</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($rows)): ?>
                        <tr>
                            <td colspan="5" style="text-align: center;">
                                Tidak ada data chat untuk tanggal yang dipilih.
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php
                        $start_number = $offset + 1;
                        foreach ($rows as $r): ?>
                            <tr>
                                <td><?php echo $start_number++; ?></td>
                                <td><?php echo esc_html(date('d M Y H:i', strtotime($r->created_at))); ?></td>
                                <td><?php echo esc_html($r->sender); ?></td>
                                <td style="max-width:400px; word-wrap: break-word;">
                                    <?php
                                    // Hapus spasi di awal pesan yang sudah tersimpan
                                    $clean_message = ltrim($r->message);
                                    echo nl2br(esc_html($clean_message));
                                    ?>
                                </td>
                                <td>
                                    <?php if ($r->user_id > 0): ?>
                                        <?php echo esc_html($r->name); ?><br>
                                        <small>📧 <?php echo esc_html($r->email); ?></small><br>
                                        <small>📱 <?php echo esc_html($r->whatsapp); ?></small>
                                    <?php else: ?>
                                        Guest User
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <!-- Pagination Navigation -->
            <?php if ($total_pages > 1): ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <?php
                        $base_url = add_query_arg([
                            'page' => 'cs-bot-ai-history',
                            'date' => $date,
                            'user_id' => $user_id
                        ], admin_url('admin.php'));

                        echo paginate_links([
                            'base' => $base_url . '%_%',
                            'format' => '&paged=%#%',
                            'current' => $current_page,
                            'total' => $total_pages,
                            'prev_text' => '&laquo; Sebelumnya',
                            'next_text' => 'Berikutnya &raquo;',
                            'mid_size' => 2,
                            'end_size' => 1
                        ]);
                        ?>
                    </div>
                </div>
            <?php endif; ?>

            <style>
                .tablenav {
                    margin: 20px 0;
                    display: flex;
                    justify-content: flex-end;
                }

                .tablenav-pages {
                    text-align: right;
                }

                .page-numbers {
                    padding: 5px 10px;
                    margin: 0 2px;
                    border: 1px solid #ccc;
                    text-decoration: none;
                }

                .page-numbers.current {
                    background: #0073aa;
                    color: white;
                    border-color: #0073aa;
                }

                .page-numbers:hover {
                    background: #f1f1f1;
                }
            </style>

            <script>
                jQuery(document).ready(function($) {
                    $('#cs-bot-export-csv').on('click', function() {
                        var date = $(this).data('date');
                        var userId = $(this).data('user-id');

                        // Show loading
                        $(this).html('<span class="spinner is-active"></span> Exporting...');
                        $(this).prop('disabled', true);

                        // Create form untuk download
                        var form = $('<form>', {
                            method: 'POST',
                            action: '<?php echo admin_url("admin-ajax.php"); ?>'
                        });

                        form.append($('<input>', {
                            type: 'hidden',
                            name: 'action',
                            value: 'cs_bot_ai_export_csv'
                        }));

                        form.append($('<input>', {
                            type: 'hidden',
                            name: 'date',
                            value: date
                        }));

                        form.append($('<input>', {
                            type: 'hidden',
                            name: 'user_id',
                            value: userId
                        }));

                        form.append($('<input>', {
                            type: 'hidden',
                            name: 'nonce',
                            value: '<?php echo wp_create_nonce("cs_bot_ai_export_nonce"); ?>'
                        }));

                        // Tambahkan form ke body dan submit
                        form.appendTo('body').submit().remove();

                        // Reset button setelah 2 detik
                        setTimeout(function() {
                            $('#cs-bot-export-csv').html('Export CSV').prop('disabled', false);
                        }, 2000);
                    });
                });
            </script>
        </div>
    <?php
    }

    public function tampilan_page()
    {
        // Simpan setting jika form disubmit
        if (isset($_POST['cs_bot_ai_tampilan_nonce']) && wp_verify_nonce($_POST['cs_bot_ai_tampilan_nonce'], 'cs_bot_ai_tampilan_save')) {
            $chat_button_color = sanitize_hex_color($_POST['chat_button_color'] ?? '#0073aa');
            $title_text_color = sanitize_hex_color($_POST['title_text_color'] ?? '#000000');
            $user_chat_gradient_start = sanitize_hex_color($_POST['user_chat_gradient_start'] ?? '#4facfe');
            $user_chat_gradient_end = sanitize_hex_color($_POST['user_chat_gradient_end'] ?? '#00f2fe');
            $header_chat_gradient_start = sanitize_hex_color($_POST['header_chat_gradient_start'] ?? '#4facfe');
            $header_chat_gradient_end = sanitize_hex_color($_POST['header_chat_gradient_end'] ?? '#00f2fe');

            update_option('cs_bot_ai_chat_button_color', $chat_button_color);
            update_option('cs_bot_ai_title_text_color', $title_text_color);
            update_option('cs_bot_ai_user_chat_gradient_start', $user_chat_gradient_start);
            update_option('cs_bot_ai_user_chat_gradient_end', $user_chat_gradient_end);
            update_option('cs_bot_ai_header_chat_gradient_start', $header_chat_gradient_start);
            update_option('cs_bot_ai_header_chat_gradient_end', $header_chat_gradient_end);

            echo '<div class="notice notice-success is-dismissible"><p>Pengaturan berhasil disimpan!</p></div>';
        }

        // Ambil value dari database
        $chat_button_color = get_option('cs_bot_ai_chat_button_color', '#0073aa');
        $title_text_color = get_option('cs_bot_ai_title_text_color', '#000000');
        $user_chat_gradient_start = get_option('cs_bot_ai_user_chat_gradient_start', '#4facfe');
        $user_chat_gradient_end = get_option('cs_bot_ai_user_chat_gradient_end', '#00f2fe');
        $header_chat_gradient_start = get_option('cs_bot_ai_header_chat_gradient_start', '#4facfe');
        $header_chat_gradient_end = get_option('cs_bot_ai_header_chat_gradient_end', '#00f2fe');

    ?>
        <div class="wrap">
            <h1>Pengaturan Tampilan CS Bot AI</h1>

            <div style="display:flex; gap:40px; align-items:flex-start;">
                <!-- Form kiri -->

                <div style="flex:1; min-width:300px;">
                    <p>Halaman ini memungkinkan Anda menyesuaikan tampilan CS Bot AI sesuai kebutuhan website. Anda dapat mengubah warna tombol chat, teks judul, header, hingga warna balon chat pengirim agar sesuai dengan identitas brand. Setiap perubahan akan langsung terlihat pada preview di samping, sehingga Anda bisa memastikan hasil desain terlihat menarik, profesional, dan konsisten dengan gaya website Anda.</p>
                    <form method="post">
                        <?php wp_nonce_field('cs_bot_ai_tampilan_save', 'cs_bot_ai_tampilan_nonce'); ?>

                        <table class="form-table">
                            <tr valign="top">
                                <th scope="row">Warna Tombol Chat</th>
                                <td colspan="2">
                                    <input type="color" name="chat_button_color" style="width: 100%;" value="<?php echo esc_attr($chat_button_color); ?>" />
                                    <p class="description">Pilih warna tombol chat yang akan muncul di website.</p>
                                </td>
                            </tr>

                            <tr valign="top">
                                <th scope="row">Warna Teks Judul</th>
                                <td colspan="2">
                                    <input type="color" name="title_text_color" style="width: 100%;" value="<?php echo esc_attr($title_text_color); ?>" />
                                    <p class="description">Pilih warna teks judul chat.</p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row">Warna Header</th>
                                <td>
                                    <input type="color" name="header_chat_gradient_start" style="width: 100%;" value="<?php echo esc_attr($header_chat_gradient_start); ?>" />
                                    <p class="description">Start</p>
                                </td>
                                <td>
                                    <input type="color" name="header_chat_gradient_end" style="width: 100%;" value="<?php echo esc_attr($header_chat_gradient_end); ?>" />
                                    <p class="description">End</p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row">Warna Chat Pengirim</th>
                                <td>
                                    <input type="color" name="user_chat_gradient_start" style="width: 100%;" value="<?php echo esc_attr($user_chat_gradient_start); ?>" />
                                    <p class="description">Start</p>
                                </td>
                                <td>
                                    <input type="color" name="user_chat_gradient_end" style="width: 100%;" value="<?php echo esc_attr($user_chat_gradient_end); ?>" />
                                    <p class="description">End</p>
                                </td>
                            </tr>
                        </table>

                        <?php submit_button('Simpan Pengaturan'); ?>
                    </form>
                </div>

                <!-- Preview kanan -->
                <div style="flex:1; min-width:400px;">
                    <h2>Preview Tampilan Chat</h2>
                    <div style="border: 1px solid #e0e0e0; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 12px rgba(0,0,0,0.1); max-width: 400px;">
                        <!-- Header -->
                        <div style="background: linear-gradient(135deg, <?php echo esc_attr($header_chat_gradient_start); ?> 0%, <?php echo esc_attr($header_chat_gradient_end); ?> 100%); padding: 20px; color: white; text-align: center;">
                            <h3 style="color: <?php echo esc_attr($title_text_color); ?>; margin: 0; font-size: 18px; font-weight: 600;">
                                DeepSeek AI Assistant
                            </h3>
                            <p style="margin: 5px 0 0 0; font-size: 12px; opacity: 0.9;">
                                Powered by DeepSeek API
                            </p>
                        </div>

                        <!-- Chat Messages -->
                        <div style="padding: 20px; background: #f8f9fa; min-height: 300px;">
                            <!-- AI Message -->
                            <div style="margin-bottom: 15px; display: flex; justify-content: flex-start;">
                                <div style="background: white; padding: 12px 16px; border-radius: 18px; border-top-left-radius: 4px; max-width: 80%; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                                    <div style="font-size: 14px; line-height: 1.4; color: #333;">
                                        Halo Kak Ahmad Shafii! Ada yang bisa saya bantu?
                                    </div>
                                    <div style="font-size: 11px; color: #999; text-align: right; margin-top: 5px;">
                                        01.08
                                    </div>
                                </div>
                            </div>

                            <!-- User Message -->
                            <div style="margin-bottom: 15px; display: flex; justify-content: flex-end;">
                                <div style="background: linear-gradient(135deg, <?php echo esc_attr($user_chat_gradient_start); ?> 0%, <?php echo esc_attr($user_chat_gradient_end); ?> 100%); padding: 12px 16px; border-radius: 18px; border-top-right-radius: 4px; max-width: 80%; color: white;">
                                    <div style="font-size: 14px; line-height: 1.4;">
                                        Pesan dari user akan muncul di sini
                                    </div>
                                    <div style="font-size: 11px; opacity: 0.8; text-align: right; margin-top: 5px;">
                                        Now
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Chat Input -->
                        <div style="padding: 15px; background: white; border-top: 1px solid #e0e0e0;">
                            <div style="display: flex; gap: 10px; align-items: center;">
                                <input type="text" placeholder="Ketik pesan Anda..." style="flex: 1; padding: 12px 16px; border: 1px solid #ddd; border-radius: 24px; font-size: 14px; outline: none;" disabled>
                                <button style="background: <?php echo esc_attr($chat_button_color); ?>; color: white; border: none; width: 40px; height: 40px; border-radius: 50%; display: flex; align-items: center; justify-content: center; cursor: not-allowed;">
                                    <i class="fas fa-paper-plane" style="font-size: 16px;"></i>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Floating Button Preview -->

                </div>
                <div class="">
                    <h3>Tombol Floating</h3>
                    <div style="position: relative; height: 100px; background: #f0f0f0; border-radius: 8px; display: flex; align-items: center; justify-content: center;">
                        <button class="cs-bot-ai-floating-btn" style="background-color: <?php echo esc_attr($chat_button_color); ?>; color:#fff; position: absolute; bottom: 20px; right: 20px;">
                            <i class="fas fa-comment-dots"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        <style>
            .cs-bot-ai-floating-btn {
                width: 60px;
                height: 60px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                cursor: pointer;
                box-shadow: 0 4px 10px rgba(0, 0, 0, 0.2);
                border: none;
                transition: all 0.3s ease;
                z-index: 1000;
            }

            .cs-bot-ai-floating-btn i {
                color: white;
                font-size: 24px;
                transition: transform 0.3s ease;
            }

            .cs-bot-ai-floating-btn:hover {
                transform: scale(1.1);
                box-shadow: 0 6px 14px rgba(0, 0, 0, 0.25);
            }

            /* Style untuk preview chat */
            .preview-chat-container {
                border: 1px solid #e0e0e0;
                border-radius: 12px;
                overflow: hidden;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
                max-width: 400px;
            }

            .preview-chat-header {
                background: linear-gradient(135deg, <?php echo esc_attr($header_chat_gradient_start); ?> 0%, <?php echo esc_attr($header_chat_gradient_end); ?> 100%);
                padding: 20px;
                color: white;
                text-align: center;
            }

            .preview-chat-messages {
                padding: 20px;
                background: #f8f9fa;
                min-height: 300px;
            }

            .preview-ai-message {
                background: white;
                padding: 12px 16px;
                border-radius: 18px;
                border-top-left-radius: 4px;
                max-width: 80%;
                box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
                margin-bottom: 15px;
            }

            .preview-user-message {
                background: linear-gradient(135deg, <?php echo esc_attr($user_chat_gradient_start); ?> 0%, <?php echo esc_attr($user_chat_gradient_end); ?> 100%);
                padding: 12px 16px;
                border-radius: 18px;
                border-top-right-radius: 4px;
                max-width: 80%;
                color: white;
                margin-bottom: 15px;
            }

            .preview-chat-input {
                padding: 15px;
                background: white;
                border-top: 1px solid #e0e0e0;
            }
        </style>
<?php
    }

    private function export_chat_csv($date, $user_id)
    {
        global $wpdb;
        $tbl = $wpdb->prefix . 'cs_bot_chats';
        $users_tbl = $wpdb->prefix . 'cs_bot_users';

        $start = $date . ' 00:00:00';
        $end = $date . ' 23:59:59';

        // Prepare query based on filters
        $where_clause = "c.created_at BETWEEN %s AND %s";
        $query_params = [$start, $end];

        if ($user_id > 0) {
            $where_clause .= " AND c.user_id = %d";
            $query_params[] = $user_id;
        }

        // Get all records for export
        $query = $wpdb->prepare(
            "SELECT c.*, u.name, u.email, u.whatsapp 
         FROM {$tbl} c 
         LEFT JOIN {$users_tbl} u ON u.id = c.user_id 
         WHERE {$where_clause} 
         ORDER BY c.created_at ASC",
            $query_params
        );
        $rows = $wpdb->get_results($query);

        // Set headers for CSV export
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=cs-bot-chat-' . $date . ($user_id ? '-user-' . $user_id : '') . '.csv');

        // Create output stream
        $output = fopen('php://output', 'w');

        // Add BOM to fix UTF-8 encoding in Excel
        fwrite($output, "\xEF\xBB\xBF");

        // Write header row
        fputcsv($output, ['id', 'session_id', 'created_at', 'sender', 'message', 'user_id', 'user_name', 'user_email', 'user_whatsapp']);

        // Write data rows
        foreach ($rows as $r) {
            // Clean message for CSV
            $clean_message = ltrim($r->message);
            $clean_message = str_replace(["\r", "\n"], "\\n", $clean_message);

            fputcsv($output, [
                $r->id,
                $r->session_id,
                $r->created_at,
                $r->sender,
                $clean_message,
                $r->user_id,
                $r->name,
                $r->email,
                $r->whatsapp
            ]);
        }

        fclose($output);
        exit;
    }
}
