<?php
if (!defined('ABSPATH')) exit;

/**
 * Unified AI API class (DeepSeek / OpenAI)
 */
class CS_Bot_AI_API
{
    private $provider;
    private $api_key;
    private $api_url;
    private $model;
    private $default_temperature = 0.7;
    private $default_max_tokens = 1000;
    private $timeout = 30;
    private $resume_interval = 600; // 10 menit
    private $settings;

    public function __construct($provider = 'deepseek')
    {
        $this->settings = get_option('cs_bot_ai_settings', []);
        $this->provider = $provider;

        if ($provider === 'deepseek') {
            $this->api_key = $this->settings['deepseek_api_key'] ?? '';
            $this->api_url = 'https://api.deepseek.com/v1/chat/completions';
            $this->model   = $this->settings['default_model'] ?? 'deepseek-chat';
        } else {
            $this->api_key = $this->settings['openai_api_key'] ?? '';
            $this->api_url = 'https://api.openai.com/v1/chat/completions';
            $this->model   = $this->settings['default_model'] ?? 'gpt-4o-mini';
        }
    }

    public function send_message($message, $history = [])
    {
        $current_time = time();
        $last_time = $history['last_time'] ?? $current_time;

        // Buat resume untuk DeepSeek jika perlu
        if ($this->provider === 'deepseek' && ($current_time - $last_time) >= $this->resume_interval && !empty($history['messages'])) {
            $resume = $this->make_resume($history['messages']);
            if ($resume) {
                $history['messages'][] = [
                    'role' => 'assistant',
                    'content' => $resume
                ];
            }
        }

        $history['last_time'] = $current_time;

        if (empty(trim($message))) {
            return $this->format_error('Pesan tidak boleh kosong', 'Empty message');
        }

        if (empty($this->api_key)) {
            return $this->format_error('Layanan AI tidak tersedia', 'API key missing');
        }

        $messages = $this->prepare_messages($message, $history['messages'] ?? []);

        $request_body = [
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => floatval($this->settings['temperature'] ?? $this->default_temperature),
            'max_tokens' => intval($this->settings['max_tokens'] ?? $this->default_max_tokens),
            'stream' => false
        ];

        $response = wp_remote_post($this->api_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode($request_body),
            'timeout' => $this->timeout,
        ]);

        if (is_wp_error($response)) {
            return $this->format_error('Terjadi kesalahan koneksi', $response->get_error_message());
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($data['error'])) {
            $error_msg = $data['error']['message'] ?? 'Unknown API error';
            return $this->format_error('Terjadi kesalahan layanan AI', $error_msg);
        }

        if (!isset($data['choices'][0]['message']['content'])) {
            return $this->format_error('Tidak ada respons dari AI', 'No response');
        }

        $ai_response = wp_kses_post($data['choices'][0]['message']['content']);

        $ai_response = str_replace('**', '', $ai_response);
        $ai_response = str_replace('###', '', $ai_response);

        // Tambahkan line break setelah tanda titik, tanda seru, atau tanda tanya
        $ai_response = preg_replace('/([.!?])\s+(?![0-9])/', "$1\n", $ai_response);

        // Tambahkan line break setelah titik dua jika bukan URL atau angka
        $ai_response = preg_replace('/:(?!\/\/)(?![0-9])\s*/', ":\n", $ai_response);

        // Hapus enter berlebih (lebih dari satu)
        $ai_response = preg_replace("/\n{2,}/", "\n", $ai_response);

        // 🔹 Ubah URL menjadi clickable (WhatsApp/HTML safe)
        $ai_response = preg_replace_callback(
            '/\b(https?:\/\/[^\s]+)/i',
            function ($matches) {
                // Untuk WhatsApp, cukup tulis link asli
                return $matches[1];
            },
            $ai_response
        );

        // ✅ Tambahkan ke history dengan format flat
        $history['messages'][] = ['role' => 'user', 'content' => $message];
        $history['messages'][] = ['role' => 'assistant', 'content' => $ai_response];

        return [
            'success' => true,
            'response' => $ai_response,
            'history' => $history
        ];
    }

    private function prepare_messages($message, $history_messages)
    {
        $messages = [];

        // 1️⃣ Ambil system_message dari pengaturan
        $system_msg = $this->settings['system_message'] ?? 'Anda adalah asisten AI yang membantu dan ramah.';
        $messages[] = [
            'role' => 'system',
            'content' => $system_msg
        ];

        // 2️⃣ Ambil instruction dari pengaturan (jika ada)
        if (!empty($history_messages) && !empty($this->settings['instruction'])) {
            $messages[] = [
                'role' => 'system',
                'content' => sanitize_textarea_field($this->settings['instruction'])
            ];
        }

        // 3️⃣ Masukkan history percakapan
        foreach ($history_messages as $msg) {
            if (isset($msg['role'], $msg['content'])) {
                $messages[] = [
                    'role' => in_array($msg['role'], ['user', 'assistant', 'system']) ? $msg['role'] : 'user',
                    'content' => sanitize_text_field($msg['content'])
                ];
            }
        }

        // 4️⃣ Tambahkan pesan user saat ini
        $messages[] = [
            'role' => 'user',
            'content' => sanitize_text_field($message)
        ];

        return $messages;
    }

    private function make_resume($history_messages)
    {
        $messages = [['role' => 'system', 'content' => 'Buat ringkasan percakapan 3-5 kalimat singkat, jelas, mudah dipahami.']];
        foreach ($history_messages as $msg) {
            if (isset($msg['role'], $msg['content'])) {
                $messages[] = ['role' => $msg['role'], 'content' => $msg['content']];
            }
        }

        $request_body = [
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => 0.5,
            'max_tokens' => 300
        ];

        if (empty($this->api_key)) return null;

        $response = wp_remote_post($this->api_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($request_body),
            'timeout' => $this->timeout,
        ]);

        if (is_wp_error($response)) return null;

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['choices'][0]['message']['content'] ?? null;
    }

    private function format_error($user_msg, $detail = '')
    {
        return ['success' => false, 'response' => $user_msg, 'error' => $detail];
    }
}
